/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.web.setup;

import com.google.common.base.Charsets;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.atlas.ApplicationProperties;
import org.apache.atlas.AtlasException;
import org.apache.atlas.ha.AtlasServerIdSelector;
import org.apache.atlas.ha.HAConfiguration;
import org.apache.atlas.setup.SetupException;
import org.apache.atlas.setup.SetupStep;
import org.apache.atlas.web.service.AtlasZookeeperSecurityProperties;
import org.apache.atlas.web.service.CuratorFactory;
import org.apache.commons.configuration.Configuration;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.BackgroundPathAndBytesable;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.stereotype.Component;

@Singleton
@Component
@Conditional(value={SetupRequired.class})
public class SetupSteps {
    private static final Logger LOG = LoggerFactory.getLogger(SetupSteps.class);
    public static final String SETUP_IN_PROGRESS_NODE = "/setup_in_progress";
    private final Set<SetupStep> setupSteps;
    private final Configuration configuration;
    private CuratorFactory curatorFactory;

    @Inject
    public SetupSteps(Set<SetupStep> steps, CuratorFactory curatorFactory, Configuration configuration) {
        this.setupSteps = steps;
        this.curatorFactory = curatorFactory;
        this.configuration = configuration;
    }

    @PostConstruct
    public void runSetup() throws SetupException {
        HAConfiguration.ZookeeperProperties zookeeperProperties = HAConfiguration.getZookeeperProperties((Configuration)this.configuration);
        InterProcessMutex lock = this.curatorFactory.lockInstance(zookeeperProperties.getZkRoot());
        try {
            LOG.info("Trying to acquire lock for running setup.");
            lock.acquire();
            LOG.info("Acquired lock for running setup.");
            this.handleSetupInProgress(this.configuration, zookeeperProperties);
            for (SetupStep step : this.setupSteps) {
                LOG.info("Running setup step: {}", (Object)step);
                step.run();
            }
            this.clearSetupInProgress(zookeeperProperties);
        }
        catch (SetupException se) {
            LOG.error("Got setup exception while trying to setup", (Throwable)se);
            throw se;
        }
        catch (Throwable e) {
            LOG.error("Error running setup steps", e);
            throw new SetupException("Error running setup steps", e);
        }
        finally {
            this.releaseLock(lock);
            this.curatorFactory.close();
        }
    }

    private void handleSetupInProgress(Configuration configuration, HAConfiguration.ZookeeperProperties zookeeperProperties) throws SetupException {
        if (this.setupInProgress(zookeeperProperties)) {
            throw new SetupException("A previous setup run may not have completed cleanly. Ensure setup can run and retry after clearing the zookeeper node at " + this.lockPath(zookeeperProperties));
        }
        this.createSetupInProgressNode(configuration, zookeeperProperties);
    }

    private void releaseLock(InterProcessMutex lock) {
        try {
            lock.release();
            LOG.info("Released lock after running setup.");
        }
        catch (Exception e) {
            LOG.error("Error releasing acquired lock.", (Throwable)e);
        }
    }

    private boolean setupInProgress(HAConfiguration.ZookeeperProperties zookeeperProperties) {
        CuratorFramework client = this.curatorFactory.clientInstance();
        String path = this.lockPath(zookeeperProperties);
        try {
            Stat lockInProgressStat = (Stat)client.checkExists().forPath(path);
            return lockInProgressStat != null;
        }
        catch (Exception e) {
            LOG.error("Error checking if path {} exists.", (Object)path, (Object)e);
            return true;
        }
    }

    private void clearSetupInProgress(HAConfiguration.ZookeeperProperties zookeeperProperties) throws SetupException {
        CuratorFramework client = this.curatorFactory.clientInstance();
        String path = this.lockPath(zookeeperProperties);
        try {
            client.delete().forPath(path);
            LOG.info("Deleted lock path after completing setup {}", (Object)path);
        }
        catch (Exception e) {
            throw new SetupException(String.format("SetupSteps.clearSetupInProgress: Failed to get Zookeeper node patH: %s", path), (Throwable)e);
        }
    }

    private String lockPath(HAConfiguration.ZookeeperProperties zookeeperProperties) {
        return zookeeperProperties.getZkRoot() + SETUP_IN_PROGRESS_NODE;
    }

    private String getServerId(Configuration configuration) {
        String serverId = configuration.getString("atlas.rest.address", "http://localhost:21000");
        try {
            serverId = AtlasServerIdSelector.selectServerId((Configuration)configuration);
        }
        catch (AtlasException e) {
            LOG.error("Could not select server id, defaulting to {}", (Object)serverId, (Object)e);
        }
        return serverId;
    }

    private void createSetupInProgressNode(Configuration configuration, HAConfiguration.ZookeeperProperties zookeeperProperties) throws SetupException {
        String serverId = this.getServerId(configuration);
        ACL acl = AtlasZookeeperSecurityProperties.parseAcl(zookeeperProperties.getAcl(), (ACL)ZooDefs.Ids.OPEN_ACL_UNSAFE.get(0));
        List<ACL> acls = Arrays.asList(acl);
        CuratorFramework client = this.curatorFactory.clientInstance();
        try {
            String path = this.lockPath(zookeeperProperties);
            ((BackgroundPathAndBytesable)client.create().withACL(acls)).forPath(path, serverId.getBytes(Charsets.UTF_8));
            LOG.info("Created lock node {}", (Object)path);
        }
        catch (Exception e) {
            throw new SetupException("Could not create lock node before running setup.", (Throwable)e);
        }
    }

    static class SetupRequired
    implements Condition {
        private static final String ATLAS_SERVER_RUN_SETUP_KEY = "atlas.server.run.setup.on.start";

        SetupRequired() {
        }

        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            try {
                Configuration configuration = ApplicationProperties.get();
                boolean shouldRunSetup = configuration.getBoolean(ATLAS_SERVER_RUN_SETUP_KEY, false);
                if (shouldRunSetup) {
                    LOG.warn("Running setup per configuration {}.", (Object)ATLAS_SERVER_RUN_SETUP_KEY);
                    return true;
                }
                LOG.info("Not running setup per configuration {}.", (Object)ATLAS_SERVER_RUN_SETUP_KEY);
            }
            catch (AtlasException e) {
                LOG.error("Unable to read config to determine if setup is needed. Not running setup.");
            }
            return false;
        }
    }
}

