/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.ColumnFamily;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.ColumnFamilyStore;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.DecoratedKey;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.OnDiskAtom;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.Row;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.RowIndexEntry;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.TreeMapBackedSortedColumns;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.CompactionController;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.CompactionInfo;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.CompactionInterruptedException;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.CompactionManager;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.LazilyCompactedRow;
import com.netflix.astyanax.shaded.org.apache.cassandra.db.compaction.OperationType;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.sstable.Component;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.sstable.Descriptor;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.sstable.SSTableReader;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.sstable.SSTableWriter;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.util.FileUtils;
import com.netflix.astyanax.shaded.org.apache.cassandra.io.util.RandomAccessReader;
import com.netflix.astyanax.shaded.org.apache.cassandra.utils.ByteBufferUtil;
import com.netflix.astyanax.shaded.org.apache.cassandra.utils.OutputHandler;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

public class Scrubber
implements Closeable {
    public final ColumnFamilyStore cfs;
    public final SSTableReader sstable;
    public final File destination;
    public final boolean skipCorrupted;
    public final boolean validateColumns;
    private final CompactionController controller;
    private final boolean isCommutative;
    private final long expectedBloomFilterSize;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final ScrubInfo scrubInfo;
    private SSTableWriter writer;
    private SSTableReader newSstable;
    private SSTableReader newInOrderSstable;
    private int goodRows;
    private int badRows;
    private int emptyRows;
    private ByteBuffer currentIndexKey;
    private ByteBuffer nextIndexKey;
    long currentRowPositionFromIndex;
    long nextRowPositionFromIndex;
    private final OutputHandler outputHandler;
    private static final Comparator<Row> rowComparator = new Comparator<Row>(){

        @Override
        public int compare(Row r1, Row r2) {
            return r1.key.compareTo(r2.key);
        }
    };
    private final SortedSet<Row> outOfOrderRows = new TreeSet<Row>(rowComparator);

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable, boolean skipCorrupted, boolean checkData) throws IOException {
        this(cfs, sstable, skipCorrupted, checkData, new OutputHandler.LogOutput(), false);
    }

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable, boolean skipCorrupted, boolean checkData, OutputHandler outputHandler, boolean isOffline) throws IOException {
        this.cfs = cfs;
        this.sstable = sstable;
        this.outputHandler = outputHandler;
        this.skipCorrupted = skipCorrupted;
        this.validateColumns = checkData;
        List<SSTableReader> toScrub = Collections.singletonList(sstable);
        this.destination = cfs.directories.getWriteableLocationAsFile(cfs.getExpectedCompactedFileSize(toScrub, OperationType.SCRUB));
        if (this.destination == null) {
            throw new IOException("disk full");
        }
        this.controller = isOffline ? new ScrubController(cfs) : new CompactionController(cfs, Collections.singleton(sstable), CompactionManager.getDefaultGcBefore(cfs));
        this.isCommutative = cfs.metadata.getDefaultValidator().isCommutative();
        boolean hasIndexFile = new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)).exists();
        if (!hasIndexFile) {
            outputHandler.warn("Missing component: " + sstable.descriptor.filenameFor(Component.PRIMARY_INDEX));
        }
        this.expectedBloomFilterSize = Math.max((long)cfs.metadata.getIndexInterval(), hasIndexFile ? SSTableReader.getApproximateKeyCount(toScrub, cfs.metadata) : 0L);
        this.dataFile = isOffline ? sstable.openDataReader() : sstable.openDataReader(CompactionManager.instance.getRateLimiter());
        this.indexFile = hasIndexFile ? RandomAccessReader.open(new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX))) : null;
        this.scrubInfo = new ScrubInfo(this.dataFile, sstable);
        this.currentRowPositionFromIndex = 0L;
        this.nextRowPositionFromIndex = 0L;
    }

    public void scrub() {
        this.outputHandler.output(String.format("Scrubbing %s (%s bytes)", this.sstable, this.dataFile.length()));
        try {
            ByteBuffer byteBuffer = this.nextIndexKey = this.indexAvailable() ? ByteBufferUtil.readWithShortLength(this.indexFile) : null;
            if (this.indexAvailable()) {
                long firstRowPositionFromIndex = RowIndexEntry.serializer.deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
                assert (firstRowPositionFromIndex == 0L) : firstRowPositionFromIndex;
            }
            this.writer = CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, this.sstable);
            DecoratedKey prevKey = null;
            while (!this.dataFile.isEOF()) {
                if (this.scrubInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.scrubInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                DecoratedKey key = null;
                long dataSize = -1L;
                try {
                    key = this.sstable.partitioner.decorateKey(ByteBufferUtil.readWithShortLength(this.dataFile));
                    if (this.sstable.descriptor.version.hasRowSizeAndColumnCount) {
                        dataSize = this.dataFile.readLong();
                        this.outputHandler.debug(String.format("row %s is %s bytes", ByteBufferUtil.bytesToHex(key.key), dataSize));
                    }
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                this.updateIndexKey();
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = -1L;
                long dataSizeFromIndex = -1L;
                if (this.currentIndexKey != null) {
                    dataStartFromIndex = this.currentRowPositionFromIndex + 2L + (long)this.currentIndexKey.remaining();
                    if (this.sstable.descriptor.version.hasRowSizeAndColumnCount) {
                        dataStartFromIndex += 8L;
                    }
                    dataSizeFromIndex = this.nextRowPositionFromIndex - dataStartFromIndex;
                }
                if (!this.sstable.descriptor.version.hasRowSizeAndColumnCount) {
                    dataSize = dataSizeFromIndex;
                    String keyName = key == null ? "(unreadable key)" : ByteBufferUtil.bytesToHex(key.key);
                    this.outputHandler.debug(String.format("row %s is %s bytes", keyName, dataSize));
                } else if (this.currentIndexKey != null) {
                    this.outputHandler.debug(String.format("Index doublecheck: row %s is %s bytes", ByteBufferUtil.bytesToHex(this.currentIndexKey), dataSizeFromIndex));
                }
                assert (this.currentIndexKey != null || !this.indexAvailable());
                this.writer.mark();
                try {
                    if (key == null) {
                        throw new IOError(new IOException("Unable to read row key from data file"));
                    }
                    if (this.currentIndexKey != null && !key.key.equals(this.currentIndexKey)) {
                        throw new IOError(new IOException(String.format("Key from data file (%s) does not match key from index file (%s)", ByteBufferUtil.bytesToHex(key.key), ByteBufferUtil.bytesToHex(this.currentIndexKey))));
                    }
                    if (dataSize > this.dataFile.length()) {
                        throw new IOError(new IOException("Impossible row size (greater than file length): " + dataSize));
                    }
                    if (this.indexFile != null && dataStart != dataStartFromIndex) {
                        this.outputHandler.warn(String.format("Data file row position %d differs from index file row position %d", dataStart, dataStartFromIndex));
                    }
                    if (this.indexFile != null && dataSize != dataSizeFromIndex) {
                        this.outputHandler.warn(String.format("Data file row size %d differs from index file row size %d", dataSize, dataSizeFromIndex));
                    }
                    SSTableIdentityIterator atoms = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataSize, this.validateColumns);
                    if (prevKey != null && prevKey.compareTo(key) > 0) {
                        this.saveOutOfOrderRow(prevKey, key, atoms);
                        continue;
                    }
                    LazilyCompactedRow compactedRow = new LazilyCompactedRow(this.controller, Collections.singletonList(atoms));
                    if (this.writer.append(compactedRow) == null) {
                        ++this.emptyRows;
                    } else {
                        ++this.goodRows;
                    }
                    prevKey = key;
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                    this.outputHandler.warn("Error reading row (stacktrace follows):", th);
                    this.writer.resetAndTruncate();
                    if (!(this.currentIndexKey == null || key != null && key.key.equals(this.currentIndexKey) && dataStart == dataStartFromIndex && dataSize == dataSizeFromIndex)) {
                        this.outputHandler.output(String.format("Retrying from row index; data is %s bytes starting at %s", dataSizeFromIndex, dataStartFromIndex));
                        key = this.sstable.partitioner.decorateKey(this.currentIndexKey);
                        try {
                            this.dataFile.seek(dataStartFromIndex);
                            SSTableIdentityIterator atoms = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataSize, this.validateColumns);
                            if (prevKey != null && prevKey.compareTo(key) > 0) {
                                this.saveOutOfOrderRow(prevKey, key, atoms);
                                continue;
                            }
                            LazilyCompactedRow compactedRow = new LazilyCompactedRow(this.controller, Collections.singletonList(atoms));
                            if (this.writer.append(compactedRow) == null) {
                                ++this.emptyRows;
                            } else {
                                ++this.goodRows;
                            }
                            prevKey = key;
                        }
                        catch (Throwable th2) {
                            this.throwIfFatal(th2);
                            this.throwIfCommutative(key, th2);
                            this.outputHandler.warn("Retry failed too. Skipping to next row (retry's stacktrace follows)", th2);
                            this.writer.resetAndTruncate();
                            ++this.badRows;
                            this.seekToNextRow();
                        }
                        continue;
                    }
                    this.throwIfCommutative(key, th);
                    this.outputHandler.warn("Row starting at position " + dataStart + " is unreadable; skipping to next");
                    ++this.badRows;
                    if (this.currentIndexKey == null) continue;
                    this.seekToNextRow();
                }
            }
            if (this.writer.getFilePointer() > 0L) {
                this.newSstable = this.writer.closeAndOpenReader(this.sstable.maxDataAge);
            }
        }
        catch (Throwable t) {
            if (this.writer != null) {
                this.writer.abort();
            }
            throw Throwables.propagate((Throwable)t);
        }
        finally {
            this.controller.close();
        }
        if (!this.outOfOrderRows.isEmpty()) {
            SSTableWriter inOrderWriter = CompactionManager.createWriter(this.cfs, this.destination, this.expectedBloomFilterSize, this.sstable);
            for (Row row : this.outOfOrderRows) {
                inOrderWriter.append(row.key, row.cf);
            }
            this.newInOrderSstable = inOrderWriter.closeAndOpenReader(this.sstable.maxDataAge);
            this.outputHandler.warn(String.format("%d out of order rows found while scrubbing %s; Those have been written (in order) to a new sstable (%s)", this.outOfOrderRows.size(), this.sstable, this.newInOrderSstable));
        }
        if (this.newSstable == null) {
            if (this.badRows > 0) {
                this.outputHandler.warn("No valid rows found while scrubbing " + this.sstable + "; it is marked for deletion now. If you want to attempt manual recovery, you can find a copy in the pre-scrub snapshot");
            } else {
                this.outputHandler.output("Scrub of " + this.sstable + " complete; looks like all " + this.emptyRows + " rows were tombstoned");
            }
        } else {
            this.outputHandler.output("Scrub of " + this.sstable + " complete: " + this.goodRows + " rows in new sstable and " + this.emptyRows + " empty (tombstoned) rows dropped");
            if (this.badRows > 0) {
                this.outputHandler.warn("Unable to recover " + this.badRows + " rows that were skipped.  You can attempt manual recovery from the pre-scrub snapshot.  You can also run nodetool repair to transfer the data from a healthy replica, if any");
            }
        }
    }

    private void updateIndexKey() {
        this.currentIndexKey = this.nextIndexKey;
        this.currentRowPositionFromIndex = this.nextRowPositionFromIndex;
        try {
            this.nextIndexKey = !this.indexAvailable() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
            this.nextRowPositionFromIndex = !this.indexAvailable() ? this.dataFile.length() : RowIndexEntry.serializer.deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
        }
        catch (Throwable th) {
            this.outputHandler.warn("Error reading index file", th);
            this.nextIndexKey = null;
            this.nextRowPositionFromIndex = this.dataFile.length();
        }
    }

    private boolean indexAvailable() {
        return this.indexFile != null && !this.indexFile.isEOF();
    }

    private void seekToNextRow() {
        while (this.nextRowPositionFromIndex < this.dataFile.length()) {
            try {
                this.dataFile.seek(this.nextRowPositionFromIndex);
                return;
            }
            catch (Throwable th) {
                this.throwIfFatal(th);
                this.outputHandler.warn(String.format("Failed to seek to next row position %d", this.nextRowPositionFromIndex), th);
                ++this.badRows;
                this.updateIndexKey();
            }
        }
    }

    private void saveOutOfOrderRow(DecoratedKey prevKey, DecoratedKey key, SSTableIdentityIterator atoms) {
        this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", key, prevKey));
        TreeMapBackedSortedColumns cf = atoms.getColumnFamily().cloneMeShallow(TreeMapBackedSortedColumns.factory, false);
        while (atoms.hasNext()) {
            OnDiskAtom atom = atoms.next();
            cf.addAtom(atom);
        }
        this.outOfOrderRows.add(new Row(key, (ColumnFamily)cf));
    }

    public SSTableReader getNewSSTable() {
        return this.newSstable;
    }

    public SSTableReader getNewInOrderSSTable() {
        return this.newInOrderSstable;
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    private void throwIfCommutative(DecoratedKey key, Throwable th) {
        if (this.isCommutative && !this.skipCorrupted) {
            this.outputHandler.warn(String.format("An error occurred while scrubbing the row with key '%s'.  Skipping corrupt rows in counter tables will result in undercounts for the affected counters (see CASSANDRA-2759 for more details), so by default the scrub will stop at this point.  If you would like to skip the row anyway and continue scrubbing, re-run the scrub with the --skip-corrupted option.", key));
            throw new IOError(th);
        }
    }

    @Override
    public void close() {
        FileUtils.closeQuietly(this.dataFile);
        FileUtils.closeQuietly(this.indexFile);
    }

    public CompactionInfo.Holder getScrubInfo() {
        return this.scrubInfo;
    }

    @VisibleForTesting
    public ScrubResult scrubWithResult() {
        this.scrub();
        return new ScrubResult(this);
    }

    public static final class ScrubResult {
        public final int goodRows;
        public final int badRows;
        public final int emptyRows;

        public ScrubResult(Scrubber scrubber) {
            this.goodRows = scrubber.goodRows;
            this.badRows = scrubber.badRows;
            this.emptyRows = scrubber.emptyRows;
        }
    }

    private static class ScrubController
    extends CompactionController {
        public ScrubController(ColumnFamilyStore cfs) {
            super(cfs, Integer.MAX_VALUE);
        }

        @Override
        public boolean shouldPurge(DecoratedKey key, long delTimestamp) {
            return false;
        }
    }

    private static class ScrubInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;

        public ScrubInfo(RandomAccessReader dataFile, SSTableReader sstable) {
            this.dataFile = dataFile;
            this.sstable = sstable;
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata, OperationType.SCRUB, this.dataFile.getFilePointer(), this.dataFile.length());
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
        }
    }
}

