/*
 * Decompiled with CFR 0.152.
 */
package org.apache.atlas.repository.audit;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.inject.Inject;
import org.apache.atlas.AtlasException;
import org.apache.atlas.EntityAuditEvent;
import org.apache.atlas.RequestContext;
import org.apache.atlas.listener.EntityChangeListener;
import org.apache.atlas.model.glossary.AtlasGlossaryTerm;
import org.apache.atlas.repository.audit.EntityAuditRepository;
import org.apache.atlas.type.AtlasEntityType;
import org.apache.atlas.type.AtlasStructType;
import org.apache.atlas.type.AtlasType;
import org.apache.atlas.type.AtlasTypeRegistry;
import org.apache.atlas.utils.AtlasPerfMetrics;
import org.apache.atlas.v1.model.instance.Referenceable;
import org.apache.atlas.v1.model.instance.Struct;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class EntityAuditListener
implements EntityChangeListener {
    private static final Logger LOG = LoggerFactory.getLogger(EntityAuditListener.class);
    private final EntityAuditRepository auditRepository;
    private final AtlasTypeRegistry typeRegistry;

    @Inject
    public EntityAuditListener(EntityAuditRepository auditRepository, AtlasTypeRegistry typeRegistry) {
        this.auditRepository = auditRepository;
        this.typeRegistry = typeRegistry;
    }

    public void onEntitiesAdded(Collection<Referenceable> entities, boolean isImport) throws AtlasException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
        ArrayList<EntityAuditEvent> events = new ArrayList<EntityAuditEvent>();
        for (Referenceable entity : entities) {
            EntityAuditEvent event = this.createEvent(entity, isImport ? EntityAuditEvent.EntityAuditAction.ENTITY_IMPORT_CREATE : EntityAuditEvent.EntityAuditAction.ENTITY_CREATE);
            events.add(event);
        }
        this.auditRepository.putEventsV1(events);
        RequestContext.get().endMetricRecord(metric);
    }

    public void onEntitiesUpdated(Collection<Referenceable> entities, boolean isImport) throws AtlasException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
        ArrayList<EntityAuditEvent> events = new ArrayList<EntityAuditEvent>();
        for (Referenceable entity : entities) {
            EntityAuditEvent event = this.createEvent(entity, isImport ? EntityAuditEvent.EntityAuditAction.ENTITY_IMPORT_UPDATE : EntityAuditEvent.EntityAuditAction.ENTITY_UPDATE);
            events.add(event);
        }
        this.auditRepository.putEventsV1(events);
        RequestContext.get().endMetricRecord(metric);
    }

    public void onTraitsAdded(Referenceable entity, Collection<? extends Struct> traits) throws AtlasException {
        if (traits != null) {
            AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
            for (Struct struct : traits) {
                EntityAuditEvent event = this.createEvent(entity, EntityAuditEvent.EntityAuditAction.TAG_ADD, "Added trait: " + AtlasType.toV1Json((Object)struct));
                this.auditRepository.putEventsV1(event);
            }
            RequestContext.get().endMetricRecord(metric);
        }
    }

    public void onTraitsDeleted(Referenceable entity, Collection<? extends Struct> traits) throws AtlasException {
        if (traits != null) {
            AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
            for (Struct struct : traits) {
                EntityAuditEvent event = this.createEvent(entity, EntityAuditEvent.EntityAuditAction.TAG_DELETE, "Deleted trait: " + struct.getTypeName());
                this.auditRepository.putEventsV1(event);
            }
            RequestContext.get().endMetricRecord(metric);
        }
    }

    public void onTraitsUpdated(Referenceable entity, Collection<? extends Struct> traits) throws AtlasException {
        if (traits != null) {
            AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
            for (Struct struct : traits) {
                EntityAuditEvent event = this.createEvent(entity, EntityAuditEvent.EntityAuditAction.TAG_UPDATE, "Updated trait: " + AtlasType.toV1Json((Object)struct));
                this.auditRepository.putEventsV1(event);
            }
            RequestContext.get().endMetricRecord(metric);
        }
    }

    public void onEntitiesDeleted(Collection<Referenceable> entities, boolean isImport) throws AtlasException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
        ArrayList<EntityAuditEvent> events = new ArrayList<EntityAuditEvent>();
        for (Referenceable entity : entities) {
            EntityAuditEvent event = this.createEvent(entity, isImport ? EntityAuditEvent.EntityAuditAction.ENTITY_IMPORT_DELETE : EntityAuditEvent.EntityAuditAction.ENTITY_DELETE, "Deleted entity");
            events.add(event);
        }
        this.auditRepository.putEventsV1(events);
        RequestContext.get().endMetricRecord(metric);
    }

    public void onTermAdded(Collection<Referenceable> entities, AtlasGlossaryTerm term) throws AtlasException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
        ArrayList<EntityAuditEvent> events = new ArrayList<EntityAuditEvent>();
        for (Referenceable entity : entities) {
            events.add(this.createEvent(entity, EntityAuditEvent.EntityAuditAction.TERM_ADD, "Added term: " + term.toAuditString()));
        }
        this.auditRepository.putEventsV1(events);
        RequestContext.get().endMetricRecord(metric);
    }

    public void onTermDeleted(Collection<Referenceable> entities, AtlasGlossaryTerm term) throws AtlasException {
        AtlasPerfMetrics.MetricRecorder metric = RequestContext.get().startMetricRecord("entityAudit");
        ArrayList<EntityAuditEvent> events = new ArrayList<EntityAuditEvent>();
        for (Referenceable entity : entities) {
            events.add(this.createEvent(entity, EntityAuditEvent.EntityAuditAction.TERM_DELETE, "Deleted term: " + term.toAuditString()));
        }
        this.auditRepository.putEventsV1(events);
        RequestContext.get().endMetricRecord(metric);
    }

    public List<EntityAuditEvent> getAuditEvents(String guid) throws AtlasException {
        return this.auditRepository.listEventsV1(guid, null, (short)10);
    }

    private EntityAuditEvent createEvent(Referenceable entity, EntityAuditEvent.EntityAuditAction action) throws AtlasException {
        String detail = this.getAuditEventDetail(entity, action);
        return this.createEvent(entity, action, detail);
    }

    private EntityAuditEvent createEvent(Referenceable entity, EntityAuditEvent.EntityAuditAction action, String details) throws AtlasException {
        return new EntityAuditEvent(entity.getId()._getId(), Long.valueOf(RequestContext.get().getRequestTime()), RequestContext.get().getUser(), action, details, entity);
    }

    private String getAuditEventDetail(Referenceable entity, EntityAuditEvent.EntityAuditAction action) throws AtlasException {
        Map<String, Object> prunedAttributes = this.pruneEntityAttributesForAudit(entity);
        String auditPrefix = EntityAuditListener.getV1AuditPrefix(action);
        String auditString = auditPrefix + AtlasType.toV1Json((Object)entity);
        byte[] auditBytes = auditString.getBytes(StandardCharsets.UTF_8);
        long auditSize = auditBytes != null ? (long)auditBytes.length : 0L;
        long auditMaxSize = this.auditRepository.repositoryMaxSize();
        if (auditMaxSize >= 0L && auditSize > auditMaxSize) {
            LOG.warn("audit record too long: entityType={}, guid={}, size={}; maxSize={}. entity attribute values not stored in audit", new Object[]{entity.getTypeName(), entity.getId()._getId(), auditSize, auditMaxSize});
            Map attrValues = entity.getValuesMap();
            entity.setValues(null);
            auditString = auditPrefix + AtlasType.toV1Json((Object)entity);
            auditBytes = auditString.getBytes(StandardCharsets.UTF_8);
            long l = auditSize = auditBytes != null ? (long)auditBytes.length : 0L;
            if (auditMaxSize >= 0L && auditSize > auditMaxSize) {
                LOG.warn("audit record still too long: entityType={}, guid={}, size={}; maxSize={}. audit will have only summary details", new Object[]{entity.getTypeName(), entity.getId()._getId(), auditSize, auditMaxSize});
                Referenceable shallowEntity = new Referenceable(entity.getId(), entity.getTypeName(), null, entity.getSystemAttributes(), null, null);
                auditString = auditPrefix + AtlasType.toJson((Object)shallowEntity);
            }
            entity.setValues(attrValues);
        }
        this.restoreEntityAttributes(entity, prunedAttributes);
        return auditString;
    }

    private Map<String, Object> pruneEntityAttributesForAudit(Referenceable entity) throws AtlasException {
        Map<String, Object> ret = null;
        Map entityAttributes = entity.getValuesMap();
        List<String> excludeAttributes = this.auditRepository.getAuditExcludeAttributes(entity.getTypeName());
        AtlasEntityType entityType = this.typeRegistry.getEntityTypeByName(entity.getTypeName());
        if (CollectionUtils.isNotEmpty(excludeAttributes) && MapUtils.isNotEmpty((Map)entityAttributes) && entityType != null) {
            for (AtlasStructType.AtlasAttribute attribute : entityType.getAllAttributes().values()) {
                String attrName = attribute.getName();
                Object attrValue = entityAttributes.get(attrName);
                if (excludeAttributes.contains(attrName)) {
                    if (ret == null) {
                        ret = new HashMap<String, Object>();
                    }
                    ret.put(attrName, attrValue);
                    entityAttributes.remove(attrName);
                    continue;
                }
                if (!attribute.isOwnedRef()) continue;
                if (attrValue instanceof Collection) {
                    for (Object arrElem : (Collection)attrValue) {
                        if (!(arrElem instanceof Referenceable)) continue;
                        ret = this.pruneAttributes(ret, (Referenceable)arrElem);
                    }
                    continue;
                }
                if (!(attrValue instanceof Referenceable)) continue;
                ret = this.pruneAttributes(ret, (Referenceable)attrValue);
            }
        }
        return ret;
    }

    private Map<String, Object> pruneAttributes(Map<String, Object> ret, Referenceable attribute) throws AtlasException {
        Referenceable attrInstance = attribute;
        Map<String, Object> prunedAttrs = this.pruneEntityAttributesForAudit(attrInstance);
        if (MapUtils.isNotEmpty(prunedAttrs)) {
            if (ret == null) {
                ret = new HashMap<String, Object>();
            }
            ret.put(attrInstance.getId()._getId(), prunedAttrs);
        }
        return ret;
    }

    private void restoreEntityAttributes(Referenceable entity, Map<String, Object> prunedAttributes) throws AtlasException {
        if (MapUtils.isEmpty(prunedAttributes)) {
            return;
        }
        AtlasEntityType entityType = this.typeRegistry.getEntityTypeByName(entity.getTypeName());
        if (entityType != null && MapUtils.isNotEmpty((Map)entityType.getAllAttributes())) {
            Map entityAttributes = entity.getValuesMap();
            for (AtlasStructType.AtlasAttribute attribute : entityType.getAllAttributes().values()) {
                String attrName = attribute.getName();
                Object attrValue = entityAttributes.get(attrName);
                if (prunedAttributes.containsKey(attrName)) {
                    entity.set(attrName, prunedAttributes.get(attrName));
                    continue;
                }
                if (!attribute.isOwnedRef()) continue;
                if (attrValue instanceof Collection) {
                    for (Object arrElem : (Collection)attrValue) {
                        if (!(arrElem instanceof Referenceable)) continue;
                        this.restoreAttributes(prunedAttributes, (Referenceable)arrElem);
                    }
                    continue;
                }
                if (!(attrValue instanceof Referenceable)) continue;
                this.restoreAttributes(prunedAttributes, (Referenceable)attrValue);
            }
        }
    }

    private void restoreAttributes(Map<String, Object> prunedAttributes, Referenceable attributeEntity) throws AtlasException {
        Object obj = prunedAttributes.get(attributeEntity.getId()._getId());
        if (obj instanceof Map) {
            this.restoreEntityAttributes(attributeEntity, (Map)obj);
        }
    }

    public static String getV1AuditPrefix(EntityAuditEvent.EntityAuditAction action) {
        String ret;
        switch (action) {
            case ENTITY_CREATE: {
                ret = "Created: ";
                break;
            }
            case ENTITY_UPDATE: {
                ret = "Updated: ";
                break;
            }
            case ENTITY_DELETE: {
                ret = "Deleted: ";
                break;
            }
            case TAG_ADD: {
                ret = "Added trait: ";
                break;
            }
            case TAG_DELETE: {
                ret = "Deleted trait: ";
                break;
            }
            case TAG_UPDATE: {
                ret = "Updated trait: ";
                break;
            }
            case ENTITY_IMPORT_CREATE: {
                ret = "Created by import: ";
                break;
            }
            case ENTITY_IMPORT_UPDATE: {
                ret = "Updated by import: ";
                break;
            }
            case ENTITY_IMPORT_DELETE: {
                ret = "Deleted by import: ";
                break;
            }
            case TERM_ADD: {
                ret = "Added term: ";
                break;
            }
            case TERM_DELETE: {
                ret = "Deleted term: ";
                break;
            }
            default: {
                ret = "Unknown: ";
            }
        }
        return ret;
    }

    public static String getV2AuditPrefix(EntityAuditEvent.EntityAuditAction action) {
        String ret;
        switch (action) {
            case ENTITY_CREATE: {
                ret = "Created: ";
                break;
            }
            case ENTITY_UPDATE: {
                ret = "Updated: ";
                break;
            }
            case ENTITY_DELETE: {
                ret = "Deleted: ";
                break;
            }
            case TAG_ADD: {
                ret = "Added classification: ";
                break;
            }
            case TAG_DELETE: {
                ret = "Deleted classification: ";
                break;
            }
            case TAG_UPDATE: {
                ret = "Updated classification: ";
                break;
            }
            case ENTITY_IMPORT_CREATE: {
                ret = "Created by import: ";
                break;
            }
            case ENTITY_IMPORT_UPDATE: {
                ret = "Updated by import: ";
                break;
            }
            case ENTITY_IMPORT_DELETE: {
                ret = "Deleted by import: ";
                break;
            }
            case TERM_ADD: {
                ret = "Added term: ";
                break;
            }
            case TERM_DELETE: {
                ret = "Deleted term: ";
                break;
            }
            default: {
                ret = "Unknown: ";
            }
        }
        return ret;
    }
}

