/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import org.apache.cassandra.cache.InstrumentingCache;
import org.apache.cassandra.cache.KeyCacheKey;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DataTracker;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.compaction.AbstractCompactedRow;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.utils.CLibrary;

public class SSTableRewriter {
    private final DataTracker dataTracker;
    private final ColumnFamilyStore cfs;
    private final long preemptiveOpenInterval;
    private final long maxAge;
    private final List<SSTableReader> finished = new ArrayList<SSTableReader>();
    private final Set<SSTableReader> rewriting;
    private final Map<Descriptor, DecoratedKey> originalStarts = new HashMap<Descriptor, DecoratedKey>();
    private final Map<Descriptor, Integer> fileDescriptors = new HashMap<Descriptor, Integer>();
    private SSTableReader currentlyOpenedEarly;
    private long currentlyOpenedEarlyAt;
    private final List<SSTableReader> finishedReaders = new ArrayList<SSTableReader>();
    private final Queue<Finished> finishedEarly = new ArrayDeque<Finished>();
    private final List<SSTableReader> discard = new ArrayList<SSTableReader>();
    private final boolean isOffline;
    private SSTableWriter writer;
    private Map<DecoratedKey, RowIndexEntry> cachedKeys = new HashMap<DecoratedKey, RowIndexEntry>();
    private State state = State.WORKING;

    public SSTableRewriter(ColumnFamilyStore cfs, Set<SSTableReader> rewriting, long maxAge, boolean isOffline) {
        this(cfs, rewriting, maxAge, isOffline, true);
    }

    public SSTableRewriter(ColumnFamilyStore cfs, Set<SSTableReader> rewriting, long maxAge, boolean isOffline, boolean shouldOpenEarly) {
        this(cfs, rewriting, maxAge, isOffline, SSTableRewriter.calculateOpenInterval(shouldOpenEarly));
    }

    @VisibleForTesting
    public SSTableRewriter(ColumnFamilyStore cfs, Set<SSTableReader> rewriting, long maxAge, boolean isOffline, long preemptiveOpenInterval) {
        this.rewriting = rewriting;
        for (SSTableReader sstable : rewriting) {
            this.originalStarts.put(sstable.descriptor, sstable.first);
            this.fileDescriptors.put(sstable.descriptor, CLibrary.getfd(sstable.getFilename()));
        }
        this.dataTracker = cfs.getDataTracker();
        this.cfs = cfs;
        this.maxAge = maxAge;
        this.isOffline = isOffline;
        this.preemptiveOpenInterval = preemptiveOpenInterval;
    }

    private static long calculateOpenInterval(boolean shouldOpenEarly) {
        long interval = (long)DatabaseDescriptor.getSSTablePreempiveOpenIntervalInMB() * 0x100000L;
        if (!shouldOpenEarly || interval < 0L) {
            interval = Long.MAX_VALUE;
        }
        return interval;
    }

    public SSTableWriter currentWriter() {
        return this.writer;
    }

    public RowIndexEntry append(AbstractCompactedRow row) {
        this.maybeReopenEarly(row.key);
        RowIndexEntry index = this.writer.append(row);
        if (!this.isOffline) {
            if (index == null) {
                this.cfs.invalidateCachedRow(row.key);
            } else {
                boolean save = false;
                for (SSTableReader reader : this.rewriting) {
                    if (reader.getCachedPosition(row.key, false) == null) continue;
                    save = true;
                    break;
                }
                if (save) {
                    this.cachedKeys.put(row.key, index);
                }
            }
        }
        return index;
    }

    public RowIndexEntry tryAppend(AbstractCompactedRow row) {
        this.writer.mark();
        try {
            return this.append(row);
        }
        catch (Throwable t) {
            this.writer.resetAndTruncate();
            throw t;
        }
    }

    private void maybeReopenEarly(DecoratedKey key) {
        if (this.writer.getFilePointer() - this.currentlyOpenedEarlyAt > this.preemptiveOpenInterval) {
            if (this.isOffline) {
                for (SSTableReader reader : this.rewriting) {
                    RowIndexEntry index = reader.getPosition(key, SSTableReader.Operator.GE);
                    CLibrary.trySkipCache((int)this.fileDescriptors.get(reader.descriptor), 0L, index == null ? 0L : index.position);
                }
            } else {
                SSTableReader reader = this.writer.openEarly(this.maxAge);
                if (reader != null) {
                    this.replaceEarlyOpenedFile(this.currentlyOpenedEarly, reader);
                    this.currentlyOpenedEarly = reader;
                    this.currentlyOpenedEarlyAt = this.writer.getFilePointer();
                    this.moveStarts(reader, reader.last, false);
                }
            }
        }
    }

    public void abort() {
        switch (this.state) {
            case ABORTED: {
                return;
            }
            case FINISHED: {
                throw new IllegalStateException("Cannot abort - changes have already been committed");
            }
        }
        this.state = State.ABORTED;
        Throwable fail = null;
        try {
            this.moveStarts(null, null, true);
        }
        catch (Throwable t) {
            fail = org.apache.cassandra.utils.Throwables.merge(fail, t);
        }
        for (SSTableReader sstable : this.finished) {
            try {
                sstable.markObsolete(null);
                sstable.selfRef().release();
            }
            catch (Throwable t) {
                fail = org.apache.cassandra.utils.Throwables.merge(fail, t);
            }
        }
        if (this.writer != null) {
            this.finishedEarly.add(new Finished(this.writer, this.currentlyOpenedEarly));
        }
        for (Finished finished : this.finishedEarly) {
            try {
                finished.writer.abort();
            }
            catch (Throwable t) {
                fail = org.apache.cassandra.utils.Throwables.merge(fail, t);
            }
            try {
                if (finished.reader == null) continue;
                this.discard.add(finished.reader);
                finished.reader.markObsolete(null);
            }
            catch (Throwable t) {
                fail = org.apache.cassandra.utils.Throwables.merge(fail, t);
            }
        }
        try {
            this.replaceWithFinishedReaders(Collections.emptyList());
        }
        catch (Throwable t) {
            fail = org.apache.cassandra.utils.Throwables.merge(fail, t);
        }
        if (fail != null) {
            throw Throwables.propagate((Throwable)fail);
        }
    }

    private void moveStarts(SSTableReader newReader, DecoratedKey lowerbound, boolean reset) {
        if (this.isOffline) {
            return;
        }
        ArrayList<SSTableReader> toReplace = new ArrayList<SSTableReader>();
        ArrayList<SSTableReader> replaceWith = new ArrayList<SSTableReader>();
        ArrayList<DecoratedKey> invalidateKeys = new ArrayList<DecoratedKey>();
        if (!reset) {
            newReader.setupKeyCache();
            invalidateKeys.addAll(this.cachedKeys.keySet());
            for (Map.Entry entry : this.cachedKeys.entrySet()) {
                newReader.cacheKey((DecoratedKey)entry.getKey(), (RowIndexEntry)entry.getValue());
            }
        }
        this.cachedKeys = new HashMap<DecoratedKey, RowIndexEntry>();
        for (SSTableReader sSTableReader : ImmutableList.copyOf(this.rewriting)) {
            SSTableReader replacement;
            SSTableReader latest = this.dataTracker.getCurrentVersion(sSTableReader);
            if (reset) {
                DecoratedKey newStart = this.originalStarts.get(sSTableReader.descriptor);
                replacement = latest.cloneWithNewStart(newStart, null);
            } else {
                if (latest.openReason == SSTableReader.OpenReason.SHADOWED || latest.first.compareTo(lowerbound) > 0) continue;
                InvalidateKeys runOnClose = new InvalidateKeys(latest, invalidateKeys);
                if (lowerbound.compareTo(latest.last) >= 0) {
                    replacement = latest.cloneAsShadowed(runOnClose);
                } else {
                    DecoratedKey newStart = latest.firstKeyBeyond(lowerbound);
                    assert (newStart != null);
                    replacement = latest.cloneWithNewStart(newStart, runOnClose);
                }
            }
            toReplace.add(latest);
            replaceWith.add(replacement);
            this.rewriting.remove(sSTableReader);
            this.rewriting.add(replacement);
        }
        this.cfs.getDataTracker().replaceWithNewInstances(toReplace, replaceWith);
    }

    private void replaceEarlyOpenedFile(SSTableReader toReplace, SSTableReader replaceWith) {
        Set<SSTableReader> toReplaceSet;
        if (this.isOffline) {
            return;
        }
        if (toReplace != null) {
            toReplace.setReplacedBy(replaceWith);
            toReplaceSet = Collections.singleton(toReplace);
        } else {
            this.dataTracker.markCompacting(Collections.singleton(replaceWith), true, this.isOffline);
            toReplaceSet = Collections.emptySet();
        }
        this.dataTracker.replaceEarlyOpenedFiles(toReplaceSet, Collections.singleton(replaceWith));
    }

    public void switchWriter(SSTableWriter newWriter) {
        if (this.writer == null) {
            this.writer = newWriter;
            return;
        }
        if (this.writer.getFilePointer() != 0L) {
            if (this.preemptiveOpenInterval == Long.MAX_VALUE) {
                SSTableReader reader = this.writer.finish(SSTableWriter.FinishType.NORMAL, this.maxAge, -1L);
                this.finishedReaders.add(reader);
            } else {
                SSTableReader reader = this.writer.finish(SSTableWriter.FinishType.EARLY, this.maxAge, -1L);
                this.replaceEarlyOpenedFile(this.currentlyOpenedEarly, reader);
                this.moveStarts(reader, reader.last, false);
                this.finishedEarly.add(new Finished(this.writer, reader));
            }
        } else {
            this.writer.abort();
        }
        this.currentlyOpenedEarly = null;
        this.currentlyOpenedEarlyAt = 0L;
        this.writer = newWriter;
    }

    public List<SSTableReader> finish() {
        return this.finish(-1L);
    }

    public List<SSTableReader> finish(long repairedAt) {
        return this.finishAndMaybeThrow(repairedAt, false, false);
    }

    @VisibleForTesting
    void finishAndThrow(boolean throwEarly) {
        this.finishAndMaybeThrow(-1L, throwEarly, !throwEarly);
    }

    private List<SSTableReader> finishAndMaybeThrow(long repairedAt, boolean throwEarly, boolean throwLate) {
        switch (this.state) {
            case ABORTED: 
            case FINISHED: {
                throw new IllegalStateException("Cannot finish - changes have already been " + this.state.toString().toLowerCase());
            }
        }
        ArrayList<SSTableReader> newReaders = new ArrayList<SSTableReader>();
        this.switchWriter(null);
        if (throwEarly) {
            throw new RuntimeException("exception thrown early in finish, for testing");
        }
        if (this.preemptiveOpenInterval == Long.MAX_VALUE) {
            this.replaceWithFinishedReaders(this.finishedReaders);
            if (throwLate) {
                throw new RuntimeException("exception thrown after all sstables finished, for testing");
            }
            return this.finishedReaders;
        }
        while (!this.finishedEarly.isEmpty()) {
            Finished f = this.finishedEarly.peek();
            if (f.writer.getFilePointer() > 0L) {
                if (f.reader != null) {
                    this.discard.add(f.reader);
                }
                SSTableReader newReader = f.writer.finish(SSTableWriter.FinishType.FINISH_EARLY, this.maxAge, repairedAt);
                if (f.reader != null) {
                    f.reader.setReplacedBy(newReader);
                }
                this.finished.add(newReader);
                newReaders.add(newReader);
            } else {
                f.writer.abort();
                assert (f.reader == null);
            }
            this.finishedEarly.poll();
        }
        if (throwLate) {
            throw new RuntimeException("exception thrown after all sstables finished, for testing");
        }
        this.replaceWithFinishedReaders(newReaders);
        this.state = State.FINISHED;
        return this.finished;
    }

    private void replaceWithFinishedReaders(List<SSTableReader> finished) {
        if (this.isOffline) {
            for (SSTableReader reader : this.discard) {
                if (!reader.isReplaced()) {
                    reader.markObsolete(null);
                }
                reader.selfRef().release();
            }
        } else {
            this.dataTracker.replaceEarlyOpenedFiles(this.discard, finished);
            this.dataTracker.unmarkCompacting(this.discard);
        }
        this.discard.clear();
    }

    private static final class Finished {
        final SSTableWriter writer;
        final SSTableReader reader;

        private Finished(SSTableWriter writer, SSTableReader reader) {
            this.writer = writer;
            this.reader = reader;
        }
    }

    private static final class InvalidateKeys
    implements Runnable {
        final List<KeyCacheKey> cacheKeys = new ArrayList<KeyCacheKey>();
        final InstrumentingCache<KeyCacheKey, ?> cache;

        private InvalidateKeys(SSTableReader reader, Collection<DecoratedKey> invalidate) {
            this.cache = reader.getKeyCache();
            if (this.cache != null) {
                for (DecoratedKey key : invalidate) {
                    this.cacheKeys.add(reader.getCacheKey(key));
                }
            }
        }

        @Override
        public void run() {
            for (KeyCacheKey key : this.cacheKeys) {
                this.cache.remove(key);
            }
        }
    }

    private static enum State {
        WORKING,
        FINISHED,
        ABORTED;

    }
}

